<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Mail\WelcomeMail;
use App\Mail\AdminNotificationMail;
use Illuminate\Support\Facades\Mail;

class AuthController extends Controller
{
    public function showRegisterForm()
    {
        session([
            'captcha_num1' => rand(1, 10),
            'captcha_num2' => rand(1, 10),
        ]);
    
        return view('auth.register', [
            'captcha_num1' => session('captcha_num1'),
            'captcha_num2' => session('captcha_num2'),
        ]);
    }
    
   public function register(Request $request)
{
    $request->validate([
        'username' => 'required|string|max:255',
        'email' => 'required|email|unique:users,email',
        'phone' => 'required|string|max:15',
        'password' => 'required|string|min:8|confirmed',
        'captcha_answer' => 'required|numeric',
    ]);

    $expectedCaptcha = session('captcha_num1') + session('captcha_num2');

    if ((int) $request->captcha_answer !== $expectedCaptcha) {
        return back()
            ->withErrors(['captcha_answer' => 'Incorrect answer to the CAPTCHA.'])
            ->withInput();
    }

    $user = User::create([
        'username' => $request->username,
        'email' => $request->email,
        'phone' => $request->phone,
        'password' => \Hash::make($request->password),
        'role' => 'user',
        'is_active' => 1,
    ]);

    return redirect()->route('login')->with('success', 'Registration successful! Please login.');
}

    


    public function showLoginForm()
    {
        return view('auth.login');
    }

    public function login(Request $request)
{
    $request->validate([
        'email' => 'required|email',
        'password' => 'required',
    ]);

    $user = User::where('email', $request->email)->first();

    if (!$user) {
        return back()
            ->withInput($request->only('email'))
            ->withErrors(['email' => 'User not found.']);
    }

    if (!$user->is_active) {
        return back()
            ->withInput($request->only('email'))
            ->withErrors(['email' => 'Your account has been deactivated. Contact admin.']);
    }

    if (Auth::attempt($request->only('email', 'password'))) {
        $role = Auth::user()->role;

        if ($role === 'admin') {
            return redirect()->route('admin.dashboard');
        } elseif ($role === 'user') {
            return redirect()->route('user.dashboard');
        }
    }

    return back()
        ->withInput($request->only('email'))
        ->withErrors(['email' => 'Invalid credentials.']);
}

    

    public function dashboard()
    {
        return view('dashboard');
    }

    public function logout()
    {
        Auth::logout();
        return redirect()->route('login')->with('success', 'Logged out successfully.');
    }
}
