<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Document;
use App\Models\User;
use Illuminate\Support\Facades\Storage;
use PhpOffice\PhpWord\IOFactory as WordIOFactory;
use PhpOffice\PhpSpreadsheet\IOFactory as ExcelIOFactory;
use Illuminate\Support\Facades\Mail;
use App\Mail\CaseAssignedMail;

class DocumentController extends Controller
{
    /**
     * Show upload form.
     */
    public function create()
    {
         return view('admin.upload-document');
    }

    /**
     * Handle document upload.
     */
    public function store(Request $request)
    {
        $requestData = $request->validate([
            'title' => 'required|string|max:255',
            'case_no' => 'required|numeric|min:1|unique:documents,case_no',
            'file' => 'required|mimes:pdf,doc,docx,xls,xlsx|max:20480',  
        ]);    
        $filePath = $request->file('file')->store('documents', 'public');
            $document = Document::create([
            'title' => $request->title,
            'case_no' => $request->case_no,
            'file_path' => $filePath,  
            'uploaded_by' => auth()->id(),
        ]);
     
        return redirect()->route('documents.list')->with('success', 'Document uploaded successfully.');
    }
    

    public function edit(Document $document)
    {
        return view('admin.edit-document', compact('document'));
    }

    /**
     * Handle document update.
     */
    public function update(Request $request, Document $document)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'case_no' => 'required',
            'file' => 'nullable|mimes:pdf,doc,docx,xls,xlsx|max:20480',
        ]);

        // ✅ Update document title
        $document->title = $request->title;

        // ✅ If a new file is uploaded, replace the old one
        if ($request->hasFile('file')) {
            Storage::disk('public')->delete($document->file_path); // Delete old file
            $filePath = $request->file('file')->store('documents', 'public');
            $document->file_path = $filePath;
        }

        $document->save();

        return redirect()->route('documents.list')->with('success', 'Document updated successfully.');
    }

    /**
     * Delete a document.
     */
    public function destroy($id)
    {
        $document = Document::findOrFail($id); // 🔍 Ensure document exists
    
        // ✅ Delete file from storage
        if ($document->file_path && Storage::disk('public')->exists($document->file_path)) {
            Storage::disk('public')->delete($document->file_path);
        }
    
        // ✅ Detach all assigned users (unassign)
        $document->users()->detach();
    
        // ✅ Get actual case_id from case_no
        $case_id = \App\Models\CaseDetail::where('case_no', $document->case_no)->value('id');
    
        // ✅ Delete all form drafts/submissions related to this case for all users
        \App\Models\FormDraft::where('case_id', $case_id)->delete();
    
        // ✅ Delete document from database
        $document->delete();
    
        return redirect()->route('documents.list')->with('success', 'Document and all related data deleted successfully.');
    }
    
    

    
    /**
     * Show assign form.
     */
   public function assignForm(Document $document)
{
    
    $assignedUserIds = $document->users->pluck('id')->toArray();

    
    $users = User::whereNotIn('id', $assignedUserIds)
                 ->where('role', '!=', 'admin')
                 ->orderBy('created_at', 'desc')
                 ->get();

   
    return view('admin.assign-document', compact('document', 'users'));
}


    /**
     * Assign document to users.
     */
   
    
    public function assign(Request $request, Document $document)
    {
        $request->validate(['users' => 'required|array']);
    
        // Attach users with assigned_at timestamp
        $assignedUsers = [];
        foreach ($request->users as $userId) {
            $assignedUsers[$userId] = ['assigned_at' => now()];
        }
    
        // Ensures existing users are not removed
        $document->users()->syncWithoutDetaching($assignedUsers);
    
        // ✅ Send email to each assigned user
       /*  foreach ($request->users as $userId) {
            $user = \App\Models\User::find($userId);
            if ($user) {
                Mail::to($user->email)->send(new CaseAssignedMail($user, $document));
            }
        } */
    
        return back()->with('success', 'Document assigned and emails sent successfully.');
    }
    
    
    /**
     * Unassign document from users.
     */
  public function unassign(Document $document, User $user)
{
    // 🔁 Detach the user from the document
    $document->users()->detach($user->id);

    $caseId = $document->case_no;
    $userId = $user->id;

    // 🧹 Delete drafts
    \App\Models\FormDraft::where('user_id', $userId)
        ->where('case_id', $caseId)
        ->delete();

    // 🧹 Delete final submitted data from all form tables
    $tables = [
        'form1_mdaat_data',
        'form2_mmt8_data',
        'form3_cdasi_data',
        'form4_cdasi_damage_data',
        'form5_data',
        'form6_pgda_data',
        'form7_mdaat_followup_data',
        'form8_followup_mmt8_data',
        'form9_cdasi_activity_data',
        'form10_cdasi_damage_data',
        'form11_data',
        'form12_pgda_data',
    ];

    foreach ($tables as $table) {
        \DB::table($table)->where('user_id', $userId)->where('case_id', $caseId)->delete();
    }

    return redirect()->back()->with('success', 'User unassigned and all related data deleted successfully.');
}

    
    

    /** 
     * List all uploaded documents.
     */
    public function index()
    {
        $documents = Document::orderBy('created_at', 'desc')->get();
        return view('admin.list-documents', compact('documents'));
    }
    
    

    public function assignPage()
{
    $documents = Document::orderBy('created_at', 'desc')->paginate(10); // Paginated List
    return view('admin.assign-page', compact('documents'));
}

public function assignedCases()
{
    $assignedCases = Document::with(['users' => function ($query) {
        $query->select('users.id', 'users.username', 'document_user.assigned_at');
    }])
    ->orderBy('created_at', 'desc')
    ->get();

    return view('admin.assigned_cases', compact('assignedCases'));
}
 

 
public function view(Document $document)
{
    if (!$document->file_path) {
        return back()->with('error', 'File not found.');
    } 

    $filePath = storage_path("app/public/{$document->file_path}");

    if (!file_exists($filePath)) {
        abort(404, 'File not found');
    }

    $extension = pathinfo($filePath, PATHINFO_EXTENSION);
    $case_no = $document->case_no;

    // ✅ Check submission status
    $userId = auth()->id();
    $isSubmitted = \App\Models\FormDraft::where('user_id', $userId)
        ->where('case_id', $case_no)
        ->where('status', 'submitted')
        ->exists();

    if (in_array($extension, ['doc', 'docx'])) {
        $phpWord = \PhpOffice\PhpWord\IOFactory::load($filePath);
        $htmlWriter = \PhpOffice\PhpWord\IOFactory::createWriter($phpWord, 'HTML');
        ob_start();
        $htmlWriter->save('php://output');
        $htmlContent = ob_get_clean();

        return view('admin.view-doc', compact('htmlContent', 'document', 'case_no', 'isSubmitted'));
    }

    elseif (in_array($extension, ['xlsx', 'xls'])) {
        $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($filePath);
        $data = $spreadsheet->getActiveSheet()->toArray();

        return view('admin.view-doc', compact('data', 'document', 'case_no', 'isSubmitted'));
    }

    elseif ($extension === 'pdf') {
        // ✅ Fix: Pass isSubmitted even if it's just for rendering PDF
        return view('admin.view-doc', compact('document', 'case_no', 'isSubmitted'));
    }

    else {
        return view('admin.view-doc', compact('document', 'case_no', 'isSubmitted'));
    }
}








}
