document.addEventListener("DOMContentLoaded", function () {

    /** ✅ Calculates total activity score including erythema, papule, periungual, alopecia, damage */
    function calculateTotalActivityScore() {
        let totalScore = 0;

        // 1. Score from activity-table
        const activityTable = document.getElementById("activity-table");
        if (activityTable) {
            activityTable.querySelectorAll("input[type='radio']:checked").forEach(radio => {
                totalScore += parseInt(radio.value) || 0;
            });
        }

        // 2. Erythema (double if papule is yes)
        const erythema = document.querySelector(`input[name="scores[erythema]"]:checked`);
        const papule = document.querySelector(`input[name="scores[papule_present]"]:checked`);

        if (erythema) {
            const erythemaVal = parseInt(erythema.value) || 0;
            if (papule && papule.value === 'yes') {
                totalScore += erythemaVal * 2;
            } else {
                totalScore += erythemaVal;
            }
        }

        // 3. Periungual
        const periungual = document.querySelector(`input[name="scores[periunguals]"]:checked`);
        if (periungual) totalScore += parseInt(periungual.value) || 0;

        // 4. Alopecia
        const alopecia = document.querySelector(`input[name="scores[alopecias]"]:checked`);
        if (alopecia) totalScore += parseInt(alopecia.value) || 0;

        // 5. Damage (Yes! Include this in activity now)
        const damage = document.querySelector(`input[name="scores[damage]"]:checked`);
        if (damage) totalScore += parseInt(damage.value) || 0;

        // 6. Display total activity score
        const totalCell = document.getElementById("total-activity-score");
        if (totalCell) {
            totalCell.textContent = totalScore;
        }

        // ✅ Also update hidden input
        const activityInput = document.getElementById("total-activity-score-hidden");
        if (activityInput) activityInput.value = totalScore;
    }

    /** ✅ Updates the readonly doubled erythema input */
    function handlePapuleScoring(erythemaName, papuleName, displayId) {
        const erythema = document.querySelector(`input[name="${erythemaName}"]:checked`);
        const papule = document.querySelector(`input[name="${papuleName}"]:checked`);
        const displayInput = document.getElementById(displayId);

        if (!erythema || !displayInput) return;

        const val = parseInt(erythema.value);
        const final = (papule && papule.value === "yes") ? val * 2 : val;

        displayInput.value = final;
    }

    /** ✅ Calculates total score for any table (e.g., damage table) */
    function calculateTotalForTable(tableId, totalCellId) {
        let total = 0;
        const table = document.getElementById(tableId);
        if (!table) return;

        table.querySelectorAll("input[type='radio']:checked").forEach(radio => {
            total += parseInt(radio.value) || 0;
        });

        const targetCell = document.getElementById(totalCellId);
        if (targetCell) {
            targetCell.textContent = total;
        }

        // ✅ Update hidden if damage
        if (totalCellId === "total-damage-score") {
            const damageInput = document.getElementById("total-damage-score-hidden");
            if (damageInput) damageInput.value = total;
        }
    }

    /** ✅ Attach listeners to radio inputs for damage/activity sections */
    function attachTableListeners(tableId, totalCellId) {
        const table = document.getElementById(tableId);
        if (!table) return;

        table.querySelectorAll("input[type='radio']").forEach(input => {
            input.addEventListener("change", function () {
                calculateTotalForTable(tableId, totalCellId);
            });
        });

        // Calculate on load
        calculateTotalForTable(tableId, totalCellId);
    }

    /** ✅ Attaches all listeners needed on load */
    function attachAllListeners() {
        // erythema + papule
        document.querySelectorAll(`input[name="scores[erythema]"], input[name="scores[papule_present]"]`)
            .forEach(input => {
                input.addEventListener("change", function () {
                    handlePapuleScoring('scores[erythema]', 'scores[papule_present]', 'erythema_final_score');
                    calculateTotalActivityScore();
                });
            });

        // radios inside activity-table
        document.querySelectorAll(`#activity-table input[type='radio']`)
            .forEach(input => {
                input.addEventListener("change", calculateTotalActivityScore);
            });

        // Periungual, Alopecia, Damage
        document.querySelectorAll(`input[name="scores[periunguals]"], input[name="scores[alopecias]"], input[name="scores[damage]"]`)
            .forEach(input => {
                input.addEventListener("change", calculateTotalActivityScore);
            });
    }

    // ✅ Attach all
    attachAllListeners();
    handlePapuleScoring('scores[erythema]', 'scores[papule_present]', 'erythema_final_score');
    calculateTotalActivityScore();

    // ✅ Damage table (initial)
    attachTableListeners("cdasi_damage-table", "total-damage-score");

    // ✅ Follow-up tables (if needed in same JS)
    attachTableListeners("cdasi_follow_up_damange-table", "total-followup-damage-score");
    attachTableListeners("cdasi_follow_up_activity-table", "total-followup-activity-score");
});



/* js for form11Cdasiactivity  */

document.addEventListener("DOMContentLoaded", function () {
    // ✅ Handles Gottron’s – Hands (Second Section) doubled score logic
    function handlePapuleScoring(erythemaName, papuleName, displayId, hiddenId = null) {
        const erythema = document.querySelector(`input[name="${erythemaName}"]:checked`);
        const papule = document.querySelector(`input[name="${papuleName}"]:checked`);

        if (!erythema) return;

        const val = parseInt(erythema.value);
        const isPapule = papule?.value === 'yes';
        const final = isPapule ? val * 2 : val;

        const displayInput = document.getElementById(displayId);
        if (displayInput) displayInput.value = final;

        if (hiddenId) {
            const hiddenInput = document.getElementById(hiddenId);
            if (hiddenInput) hiddenInput.value = final;
        }

        // Trigger follow-up activity recalculation
        calculateTotalForTable("cdasi_follow_up_activity-table", "total-followup-activity-score", [
            displayId,
            "scores[periunguals]",
            "scores[alopecias]",
            "scores[damage_2]"
        ]);
    }

    function attachListeners(erythemaName, papuleName, displayId, hiddenId = null) {
        document.querySelectorAll(`input[name="${erythemaName}"], input[name="${papuleName}"]`)
            .forEach(el => el.addEventListener('change', () =>
                handlePapuleScoring(erythemaName, papuleName, displayId, hiddenId)
            ));
    }

    // ✅ Total calculator for table + extra fields
    function calculateTotalForTable(tableId, totalCellId, extraFields = []) {
        let totalScore = 0;
        let table = document.getElementById(tableId);

        if (table) {
            table.querySelectorAll("input[type='radio']:checked").forEach(radio => {
                totalScore += parseInt(radio.value) || 0;
            });
        }

        // Add from final Gottron score and other additional fields
        extraFields.forEach(fieldName => {
            const field = document.querySelector(`[name="${fieldName}"]:checked`);
            if (field) totalScore += parseInt(field.value) || 0;

            const readonly = document.getElementById(fieldName);
            if (readonly) totalScore += parseInt(readonly.value) || 0;
        });

        const totalScoreCell = document.getElementById(totalCellId);
        if (totalScoreCell) totalScoreCell.textContent = totalScore;

        // ✅ Update corresponding hidden input
        if (totalCellId === "total-followup-activity-score") {
            const activityInput = document.getElementById("total-followup-activity-score-hidden");
            if (activityInput) activityInput.value = totalScore;
        }

        if (totalCellId === "total-followup-damage-score") {
            const damageInput = document.getElementById("total-followup-damage-score-hidden");
            if (damageInput) damageInput.value = totalScore;
        }
    }

    // ✅ Attach change listeners and calculate on load
    function attachEventListeners(tableId, totalCellId, extraFields = []) {
        let elements = document.querySelectorAll(`#${tableId} input[type='radio']`);
        extraFields.forEach(field => {
            elements = [...elements, ...document.querySelectorAll(`input[name="${field}"]`)];
        });

        elements.forEach(el => {
            el.addEventListener("change", function () {
                calculateTotalForTable(tableId, totalCellId, extraFields);
            });
        });

        calculateTotalForTable(tableId, totalCellId, extraFields);
    }

    // ✅ Initial Form Calculations
    attachEventListeners("cdasi_follow_up_activity-table", "total-followup-activity-score", [
        "erythema_final_score_2",
        "scores[periunguals]",
        "scores[alopecias]",
        "scores[damage_2]"
    ]);

    attachEventListeners("cdasi_follow_up_damange-table", "total-followup-damage-score");

    // ✅ Attach listeners for Gottron’s – Hands (Second Section)
    attachListeners('scores[erythema_2]', 'scores[papule_present_2]', 'erythema_final_score_2', 'final-erythema-score-2');
    handlePapuleScoring('scores[erythema_2]', 'scores[papule_present_2]', 'erythema_final_score_2', 'final-erythema-score-2');
});




 